<?php
// $Id: ec_store.localization.inc,v 1.2.2.4 2009/05/31 04:45:54 gordon Exp $

/**
 * @file
 * All functions that are related to regional settings
 */

// -------------------------------------------------------------------------
// Settings

/**
 * Set all Location settings, like the used physical
 * dimenstions and currency format
 *
 * @return
 *   Array. The form settings array.
 */
function ec_store_location_settings_form() {
  $countries = _ec_store_location_countries();
  $form['ec_country'] = array(
    '#type' => 'select',
    '#title' => t('Store location'),
    '#default_value' => variable_get('ec_country', 'us'),
    '#options' => $countries,
    '#multiple' => FALSE,
    '#description' => 'Default country location.',
  );

  // Physical dimensions
  $form['dimensions'] = array(
    '#collapsible'    => TRUE,
    '#title'          => t('Physical Dimensions'),
    '#tree'           => FALSE,
    '#type'           => 'fieldset'
  );
  $form['dimensions']['ec_measure_weight'] = array(
    '#default_value'  => variable_get('ec_measure_weight', 'LB'),
    '#description'    => t('Default unit measure of weight.'),
    '#options'        => _ec_store_location_measures('weight'),
    '#title'          => t('Weight'),
    '#type'           => 'select'
  );
  $form['dimensions']['ec_measure_length'] = array(
    '#default_value'  => variable_get('ec_measure_length', 'IN'),
    '#description'    => t('Default unit measure of length.'),
    '#options'        => _ec_store_location_measures('length'),
    '#title'          => t('Length'),
    '#type'           => 'select'
  );
  $form['dimensions']['ec_measure_area'] = array(
    '#default_value'  => variable_get('ec_measure_area', 'IN2'),
    '#description'    => t('Default unit measure of area.'),
    '#options'        => _ec_store_location_measures('area'),
    '#title'          => t('Area'),
    '#type'           => 'select'
  );
  $form['dimensions']['ec_measure_volume'] = array(
    '#default_value'  => variable_get('ec_measure_volume', 'IN3'),
    '#description'    => t('Default unit measure of volume.'),
    '#options'        =>  _ec_store_location_measures('volume'),
    '#title'          => t('Volume'),
    '#type'           => 'select'
  );

  // Price format settings
  $form['price_format'] = array(
    '#collapsible'    => TRUE,
    '#title'          => t('Price Formatting'),
    '#tree'           => FALSE,
    '#type'           => 'fieldset',
  );
  $form['price_format']['payment_symbol'] = array(
    '#type'           => 'textfield',
    '#title'          => t('Currency Symbol'),
    '#default_value'  => variable_get('payment_symbol', '$'),
    '#size'           => 3,
    '#maxlength'      => 5,
    '#description'    => t('Enter the currency symbol you wish to associate with your price. This will be displayed in front of the price. Default is the dollar symbol.')
  );
  $form['price_format']['payment_symbol_position'] = array(
    '#type'           => 'radios',
    '#title'          => t('Currency Symbol'),
    '#default_value'  => variable_get('payment_symbol_position', 1),
    '#options'        => array(t('Right'), t('Left')),
    '#description'    => t('This option places the currency symbol of the left or right side of the price.')
  );
  $form['price_format']['payment_thousands'] = array(
    '#type'           => 'textfield',
    '#title'          => t('Thousands separator'),
    '#default_value'  => variable_get('payment_thousands', ','),
    '#size'           => 3,
    '#maxlength'      => 5,
    '#description'    => t('Enter the sign for the thousands separator.')
  );
  $form['price_format']['payment_decimal'] = array(
    '#type'           => 'textfield',
    '#title'          => t('Decimal separator'),
    '#default_value'  => variable_get('payment_decimal', '.'),
    '#size'           => 3,
    '#maxlength'      => 5,
    '#description'    => t('Enter the sign to seperate real numbers from floating numbers.')
  );
  $form['price_format']['payment_decimal_places'] = array(
    '#type'           => 'textfield',
    '#title'          => t('Number of places after the decimal separator'),
    '#default_value'  => variable_get('payment_decimal_places', 2),
    '#size'           => 3,
    '#maxlength'      => 5,
    '#description'    => t('How many slots are needed after the decimal?')
  );

  return system_settings_form($form);
}

// -------------------------------------------------------------------------
// Big nasty arrays

/**
 * Build a list of Countries.
 *
 * @return
 *   Array, All countries names and codes
 */
function _ec_store_location_countries() {
  static $countries;
  if (empty($countries)) {
    $countries = array(
      'AD' => t('Andorra'),
      'AC' => t('Ascension Island'),
      'AE' => t('United Arab Emirates'),
      'AF' => t('Afghanistan'),
      'AG' => t('Antigua and Barbuda'),
      'AI' => t('Anguilla'),
      'AL' => t('Albania'),
      'AM' => t('Armenia'),
      'AN' => t('Netherlands Antilles'),
      'AO' => t('Angola'),
      'AQ' => t('Antarctica'),
      'AR' => t('Argentina'),
      'AS' => t('American Samoa'),
      'AT' => t('Austria'),
      'AU' => t('Australia'),
      'AW' => t('Aruba'),
      'AX' => t('Aland Islands'),
      'AZ' => t('Azerbaijan'),
      'BA' => t('Bosnia and Herzegovina'),
      'BB' => t('Barbados'),
      'BD' => t('Bangladesh'),
      'BE' => t('Belgium'),
      'BF' => t('Burkina Faso'),
      'BG' => t('Bulgaria'),
      'BH' => t('Bahrain'),
      'BI' => t('Burundi'),
      'BJ' => t('Benin'),
      'BM' => t('Bermuda'),
      'BN' => t('Brunei Darussalam'),
      'BO' => t('Bolivia'),
      'BR' => t('Brazil'),
      'BS' => t('Bahamas'),
      'BT' => t('Bhutan'),
      'BV' => t('Bouvet Island'),
      'BW' => t('Botswana'),
      'BY' => t('Belarus'),
      'BZ' => t('Belize'),
      'CA' => t('Canada'),
      'CC' => t('Cocos (Keeling) Islands'),
      'CD' => t('Democratic Republic of the Congo (formerly Zaire)'),
      'CF' => t('Central African Republic'),
      'CG' => t('Congo (Republic of the Congo)'),
      'CH' => t('Switzerland (Confoederatio Helvetica)'),
      'CI' => t('Cote d\'Ivoire (Ivory Coast)'),
      'CK' => t('Cook Islands'),
      'CL' => t('Chile'),
      'CM' => t('Cameroon'),
      'CN' => t('China'),
      'CO' => t('Colombia'),
      'CR' => t('Costa Rica'),
      'CS' => t('Serbia and Montenegro'),
      'CU' => t('Cuba'),
      'CV' => t('Cape Verde'),
      'CX' => t('Christmas Island'),
      'CY' => t('Cyprus'),
      'CZ' => t('Czech Republic'),
      'DE' => t('Germany (Deutschland)'),
      'DJ' => t('Djibouti'),
      'DK' => t('Denmark'),
      'DM' => t('Dominica'),
      'DO' => t('Dominican Republic'),
      'DZ' => t('Algeria'),
      'EC' => t('Ecuador'),
      'EE' => t('Estonia'),
      'EG' => t('Egypt'),
      'EH' => t('Western Sahara (formerly Spanish Sahara)'),
      'ER' => t('Eritrea'),
      'ES' => t('Spain (Espana)'),
      'ET' => t('Ethiopia'),
      'FI' => t('Finland'),
      'FJ' => t('Fiji'),
      'FK' => t('Falkland Islands'),
      'FM' => t('Federated States of Micronesia'),
      'FO' => t('Faroe Islands'),
      'FR' => t('France'),
      'GA' => t('Gabon'),
      'GB' => t('United Kingdom'),
      'GD' => t('Grenada'),
      'GE' => t('Georgia'),
      'GF' => t('French Guiana'),
      'GH' => t('Ghana'),
      'GI' => t('Gibraltar'),
      'GL' => t('Greenland'),
      'GM' => t('Gambia'),
      'GN' => t('Guinea'),
      'GP' => t('Guadeloupe'),
      'GQ' => t('Equatorial Guinea'),
      'GR' => t('Greece'),
      'GS' => t('South Georgia and the South Sandwich Islands'),
      'GT' => t('Guatemala'),
      'GU' => t('Guam'),
      'GW' => t('Guinea-Bissau'),
      'GY' => t('Guyana'),
      'HK' => t('Hong Kong'),
      'HM' => t('Heard Island and McDonald Islands'),
      'HN' => t('Honduras'),
      'HR' => t('Croatia (Hrvatska)'),
      'HT' => t('Haiti'),
      'HU' => t('Hungary'),
      'ID' => t('Indonesia'),
      'IE' => t('Ireland'),
      'IL' => t('Israel'),
      'IN' => t('India'),
      'IO' => t('British Indian Ocean Territory (including Diego Garcia)'),
      'IQ' => t('Iraq'),
      'IR' => t('Iran'),
      'IS' => t('Iceland'),
      'IT' => t('Italy'),
      'JM' => t('Jamaica'),
      'JO' => t('Jordan'),
      'JP' => t('Japan'),
      'KE' => t('Kenya'),
      'KG' => t('Kyrgyzstan'),
      'KH' => t('Cambodia'),
      'KI' => t('Kiribati'),
      'KM' => t('Comoros'),
      'KN' => t('Saint Kitts and Nevis'),
      'KP' => t('North Korea'),
      'KR' => t('South Korea'),
      'KW' => t('Kuwait'),
      'KY' => t('Cayman Islands'),
      'KZ' => t('Kazakhstan'),
      'LA' => t('Laos'),
      'LB' => t('Lebanon'),
      'LC' => t('Saint Lucia'),
      'LI' => t('Liechtenstein'),
      'LK' => t('Sri Lanka'),
      'LR' => t('Liberia'),
      'LS' => t('Lesotho'),
      'LT' => t('Lithuania'),
      'LU' => t('Luxembourg'),
      'LV' => t('Latvia'),
      'LY' => t('Libya'),
      'MA' => t('Morocco'),
      'MC' => t('Monaco'),
      'MD' => t('Moldova'),
      'MG' => t('Madagascar'),
      'MH' => t('Marshall Islands'),
      'MK' => t('Former Yugoslav Republic of Macedonia'),
      'ML' => t('Mali'),
      'MM' => t('Myanmar (Burma)'),
      'MN' => t('Mongolia'),
      'MO' => t('Macao (Macau)'),
      'MP' => t('Northern Mariana Islands'),
      'MQ' => t('Martinique'),
      'MR' => t('Mauritania'),
      'MS' => t('Montserrat'),
      'MT' => t('Malta'),
      'MU' => t('Mauritius'),
      'MV' => t('Maldives'),
      'MW' => t('Malawi'),
      'MX' => t('Mexico'),
      'MY' => t('Malaysia'),
      'MZ' => t('Mozambique'),
      'NA' => t('Namibia'),
      'NC' => t('New Caledonia'),
      'NE' => t('Niger'),
      'NF' => t('Norfolk Island'),
      'NG' => t('Nigeria'),
      'NI' => t('Nicaragua'),
      'NL' => t('Netherlands'),
      'NO' => t('Norway'),
      'NP' => t('Nepal'),
      'NR' => t('Nauru'),
      'NU' => t('Niue'),
      'NZ' => t('New Zealand'),
      'OM' => t('Oman'),
      'PA' => t('Panama'),
      'PE' => t('Peru'),
      'PF' => t('French Polynesia'),
      'PG' => t('Papua New Guinea'),
      'PH' => t('Philippines'),
      'PK' => t('Pakistan'),
      'PL' => t('Poland'),
      'PM' => t('Saint-Pierre and Miquelon'),
      'PN' => t('Pitcairn Islands'),
      'PR' => t('Puerto Rico'),
      'PS' => t('Palestinian Territories (West Bank and Gaza Strip)'),
      'PT' => t('Portugal'),
      'PW' => t('Palau'),
      'PY' => t('Paraguay'),
      'QA' => t('Qatar'),
      'RE' => t('Reunion'),
      'RO' => t('Romania'),
      'RU' => t('Russia'),
      'RW' => t('Rwanda'),
      'SA' => t('Saudi Arabia'),
      'SB' => t('Solomon Islands'),
      'SC' => t('Seychelles'),
      'SD' => t('Sudan'),
      'SE' => t('Sweden'),
      'SG' => t('Singapore'),
      'SH' => t('Saint Helena'),
      'SI' => t('Slovenia'),
      'SJ' => t('Svalbard and Jan Mayen Islands'),
      'SK' => t('Slovakia'),
      'SL' => t('Sierra Leone'),
      'SM' => t('San Marino'),
      'SN' => t('Senegal'),
      'SO' => t('Somalia'),
      'SR' => t('Suriname'),
      'ST' => t('Sao Tome and Principe'),
      'SV' => t('El Salvador'),
      'SW' => t('Sweden'),
      'SY' => t('Syria'),
      'SZ' => t('Swaziland'),
      'TC' => t('Turks and Caicos Islands'),
      'TD' => t('Chad (Tchad)'),
      'TF' => t('French Southern Territories'),
      'TG' => t('Togo'),
      'TH' => t('Thailand'),
      'TJ' => t('Tajikistan'),
      'TK' => t('Tokelau'),
      'TL' => t('Timor-Leste (East Timor)'),
      'TM' => t('Turkmenistan'),
      'TN' => t('Tunisia'),
      'TO' => t('Tonga'),
      'TR' => t('Turkey'),
      'TT' => t('Trinidad and Tobago'),
      'TV' => t('Tuvalu'),
      'TW' => t('Taiwan'),
      'TZ' => t('Tanzania'),
      'UA' => t('Ukraine'),
      'UG' => t('Uganda'),
      //'UK' => t('United Kingdom'),
      'UM' => t('United States Minor Outlying Islands'),
      'US' => t('United States'),
      'UY' => t('Uruguay'),
      'UZ' => t('Uzbekistan'),
      'VA' => t('Vatican City'),
      'VC' => t('Saint Vincent and the Grenadines'),
      'VE' => t('Venezuela'),
      'VG' => t('British Virgin Islands'),
      'VI' => t('U.S. Virgin Islands'),
      'VN' => t('Vietnam'),
      'VU' => t('Vanuatu'),
      'Wf' => t('Wallis and Futuna'),
      'WS' => t('Samoa'),
      'YE' => t('Yemen'),
      'YT' => t('Mayotte'),
      'ZA' => t('South Africa'),
      'ZM' => t('Zambia'),
      'ZW' => t('Zimbabwe')
    );
    natcasesort($countries);
  }
  return $countries;
}

/**
 * Get the default mesaures and currency values
 * for a given country
 *
 * @param $country
 *   String. The ISO country 2-digit code
 * @return
 *   Array. Physical dimension and currency formating options
 */
function _ec_store_location_default($country) {
  if ($country == 'AU') { // Australia
    return array(
      // Measures
      'weight'                  => 'KG',
      'length'                  => 'M',
      'area'                    => 'M2',
      'volume'                  => 'M3',
      // Geo
      'state_name'              => t('State'),
      'zip'                     => TRUE,
      // Price format
      'payment_symbol'          => '$',
      'payment_symbol_position' => 1, // Left
      'payment_thousands'       => ',',
      'payment_decimal'         => '.',
      'payment_decimal_places'  => 2,
    );
  }
  elseif ($country == 'BR') { // Brazil
    return array(
      // Measures
      'weight'                  => 'KG',
      'length'                  => 'M',
      'area'                    => 'M2',
      'volume'                  => 'M3',
      // Geo
      'state_name'              => t('State'),
      'zip'                     => TRUE,
      // Price format
      'payment_symbol'          => 'R$',
      'payment_symbol_position' => 1, // Left
      'payment_thousands'       => '.',
      'payment_decimal'         => ',',
      'payment_decimal_places'  => 2,
    );
  }
  elseif ($country == 'CA') { // Canada
    return array(
      // Measures
      'weight'                  => 'KG',
      'length'                  => 'M',
      'area'                    => 'M2',
      'volume'                  => 'M3',
      // Geo
      'state_name'              => t('Province'),
      'zip'                     => TRUE,
      // Price format
      'payment_symbol'          => '$',
      'payment_symbol_position' => 1, // Left
      'payment_thousands'       => ',',
      'payment_decimal'         => '.',
      'payment_decimal_places'  => 2,
    );
  }
  elseif ($country == 'EE') { // Estonia
    return array(
      // Measures
      'weight'                  => 'KG',
      'length'                  => 'M',
      'area'                    => 'M2',
      'volume'                  => 'M3',
      // Geo
      'state_name'              => t('State'),
      'zip'                     => TRUE,
      // Price format
      'payment_symbol'          => ' kr', // I put a space before currency on purpose, since it must be padded with a space in the end, like "29.95 kr"
      'payment_symbol_position' => 2, // Right
      'payment_thousands'       => ' ',
      'payment_decimal'         => '.',
      'payment_decimal_places'  => 2,
    );
  }
  elseif ($country == 'IE') { // Ireland
    return array(
      // Measures
      'weight'                    => 'KG',
      'length'                    => 'M',
      'area'                      => 'M2',
      'volume'                    => 'M3',
      // Geo
      'state_name'                => t('County'),
      'zip'                       => FALSE, // No zip codes in Ireland (yet!)
      // Price format
      'ec_payment_symbol'            => '€',
      'ec_payment_symbol_position'   => 1, // Left
      'ec_payment_thousands'         => ',',
      'ec_payment_decimal'           => '.',
      'ec_payment_decimal_places'    => 2,
    );
  }
  elseif ($country == 'RU') { // Russia
    return array(
    // Measures
    'weight'                    => 'KG',
    'length'                    => 'M',
    'area'                      => 'M2',
    'volume'                    => 'M3',
    // Geo
    'state_name'                => t('State'),
    'zip'                       => TRUE,
    // Price format
    'payment_symbol'            => ' руб.',     // I put a space before currency on purpose, since it must be padded with a space in the end, like "29.95 руб."
    'payment_symbol_position'   => 2,           // Right
    'payment_thousands'         => ' ',
    'payment_decimal'           => ',',
    'payment_decimal_places'    => 2,
    );
  }
  elseif ($country == 'UA') { // Ukraine
    return array(
      // Measures
      'weight'                    => 'KG',
      'length'                    => 'M',
      'area'                      => 'M2',
      'volume'                    => 'M3',
      // Geo
      'state_name'                => t('State'),  // the same format is used all over Ukraine
      'zip'                       => TRUE,
      // Price format
      'payment_symbol'            => ' грн.',     // I put a space before currency on purpose, since it must be padded with a space in the end, like "29.95 грн."
      'payment_symbol_position'   => 2,           // Right
      'payment_thousands'         => ' ',
      'payment_decimal'           => ',',
      'payment_decimal_places'    => 2,
    );
  }
  elseif ($country == 'UK' || $country == 'GB') { // United Kingdom
    return array(
      // Measures
      'weight'                    => 'KG',
      'length'                    => 'M',
      'area'                      => 'M2',
      'volume'                    => 'M3',
      // Geo
      'state_name'                => t('County'),
      'zip'                       => TRUE, // Although they're called 'Post Code'
      // Price format
      'payment_symbol'            => '£',
      'payment_symbol_position'   => 1, // Left
      'payment_thousands'         => ',',
      'payment_decimal'           => '.',
      'payment_decimal_places'    => 2,
    );
  }
  elseif ($country == 'US') { // United States
    return array(
      // Measures
      'weight'                  => 'LB',
      'length'                  => 'IN',
      'area'                    => 'IN2',
      'volume'                  => 'IN3',
      // Geo
      'state_name'              => t('State'),
      'zip'                     => TRUE,
      // Price format
      'payment_symbol'          => '$',
      'payment_symbol_position' => 1,           // Left
      'payment_thousands'       => ',',
      'payment_decimal'         => '.',
      'payment_decimal_places'  => 2,
    );
  }
  // In case there is no defined Country settings
  return FALSE;
}

/**
 * List all types of physical measures
 *
 * @return
 *   array keyed on
 *    'name', String, the measure name
 *    'factor', conversion factor to the default (eg. to kilograms or metres)
 */
function _ec_store_location_measures($type = NULL) {
  $measures = array(
    'LB'  => array('type' => 'weight', 'name' => t('Pounds'), 'factor' => 0.45359237),
    'G'   => array('type' => 'weight', 'name' => t('Grams'), 'factor' => 0.001),
    'KG'  => array('type' => 'weight', 'name' => t('Kilograms'), 'factor' => 1),
    'OZ'  => array('type' => 'weight', 'name' => t('Ounces'), 'factor' => 0.0283495231),

    'IN'  => array('type' => 'length', 'name' => t('Inches'), 'factor' => 0.0254),
    'FT'  => array('type' => 'length', 'name' => t('Feet'), 'factor' => 0.3048),
    'CM'  => array('type' => 'length', 'name' => t('Centimetres'), 'factor' => 0.01),
    'M'   => array('type' => 'length', 'name' => t('Metres'), 'factor' => 1),

    'IN2' => array('type' => 'area', 'name' => t('Square Inches'), 'factor' => 0.0254 ^ 2),
    'FT2' => array('type' => 'area', 'name' => t('Square Feet'), 'factor' => 0.3048 ^ 2),
    'CM2' => array('type' => 'area', 'name' => t('Square Centimetres'), 'factor' => 0.01 ^ 2),
    'M2'  => array('type' => 'area', 'name' => t('Square Metres'), 'factor' => 1),

    'IN3' => array('type' => 'volume', 'name' => t('Cubic Inches'), 'factor' => 0.0254 ^ 3),
    'FT3' => array('type' => 'volume', 'name' => t('Cubic Feet'), 'factor' => 0.3048 ^ 3),
    'CM3' => array('type' => 'volume', 'name' => t('Cubic Centimetres'), 'factor' => 0.01 ^ 3),
    'M3'  => array('type' => 'volume', 'name' => t('Cubic Metres'), 'factor' => 1),
  );

  if ($type) {
    $list = array();
    foreach ($measures as $measure => $data) {
      if ($data['type'] == $type) {
        $list[$measure] = $data['name'];
      }
    }
    return $list;
  }
  else {
    return $measures;
  }
}

/**
 * Convert some measure units to another. All units are converted
 * to a central unit and them converted again to the destination units.
 *
 * All weight is converted to kilograms and then to the destination.
 * All distance is converted to meters and then to the destination.
 * All area is converted to square meters and then to the destination.
 * All volume is converted to cubic meters and then to the destination.
 *
 * @param $value
 *   Number. The value in the old unit
 * @param $from
 *   String. The unit code that the value is origined
 * @param $to
 *   String. The unit code that the value should be converted to
 * @return
 *   Number. The converted value
 */
function _ec_store_location_measure_convert($value, $from, $to) {
  if ($from == $to) {
    return $value;
  }
  $default = FALSE;
  $measures = _ec_store_location_measures();
  if (!isset($measures[$from]) || !isset($measures[$to]) || $measures[$from]['type'] != $measures[$to]['type']) {
    // Something is screwy.
    watchdog('store', 'e-Commerce failed to convert '. $measures[$from] .' to '. $measures[$to] .'.', WATCHDOG_ERROR);
    return FALSE;
  }
  return $value * $measures[$from]['factor'] / $measures[$to]['factor'];
}

/**
 * Return an array of all states possible.
 */
function _ec_store_location_states_all() {
  foreach (array_keys(_ec_store_location_countries()) as $country) {
    $states[$country] = _ec_store_location_states($country);
  }
  return $states;
}

/**
 * Get available states. If $country is passed, then return an array
 * of states for that country. If nothing is passed then return
 * an array of countrys who have states, where each country contains
 * an array of states. If the requested country is not configured then
 * return an empty array.
 *
 * This function used to return all available states. This function is
 * now simpler to avoid overhead from newly added state lists. Use
 * _ec_store_location_states_all() to get all available states.
 *
 */
function _ec_store_location_states($country) {

  // Store any previously requested state list.
  static $states = array();
  if (isset($states[$country])) {
    return $states[$country];
  }

  // Only set new states on the array if they've been
  // requested, this avoids excessive calls to t().
  switch ($country) {
    case 'AU':
      $states[$country] = array(
        'ACT' => 'Australian Capital Territory',
        'NSW' => 'New South Wales',
        'NT'  => 'Northern Territory',
        'QLD' => 'Queensland',
        'SA'  => 'South Australia',
        'TAS' => 'Tasmania',
        'VIC' => 'Victoria',
        'WA'  => 'Western Australia',
      );
      break;
    case 'BR':
      $states[$country] = array(
        'AC' => 'Acre',
        'AL' => 'Alagoas',
        'AP' => 'Amapá',
        'AM' => 'Amazonas',
        'BA' => 'Bahia',
        'CE' => 'Ceará',
        'DF' => 'Distrito Federal',
        'ES' => 'Espírito Santo',
        'GO' => 'Goiás',
        'MA' => 'Maranhão',
        'MT' => 'Mato Grosso',
        'MS' => 'Mato Grosso do Sul',
        'MG' => 'Minas Gerais',
        'PA' => 'Pará',
        'PB' => 'Paraíba',
        'PR' => 'Paraná',
        'PE' => 'Pernambuco',
        'PI' => 'Piauí',
        'RJ' => 'Rio de Janeiro',
        'RN' => 'Rio Grande do Norte',
        'RS' => 'Rio Grande do Sul',
        'RO' => 'Rondônia',
        'RR' => 'Roraima',
        'SC' => 'Santa Catarina',
        'SP' => 'São Paulo',
        'SE' => 'Sergipe',
        'TO' => 'Tocantins',
      );
      break;
    case 'FR':
      $states[$country] = array(
        'A67' => t('Bas-Rhin - Alsace'),
        'A68' => t('Haut-Rhin - Alsace'),
        'B24' => t('Dordogne - Aquitaine'),
        'B33' => t('Gironde - Aquitaine'),
        'B40' => t('Landes - Aquitaine'),
        'B47' => t('Lot-et-Garonne - Aquitaine'),
        'B64' => t('Pyrenees-Atlantiques - Aquitaine'),
        'B79' => t('Deux-Sevres - Aquitaine'),
        'C03' => t('Allier - Auvergne'),
        'C15' => t('Cantal - Auvergne'),
        'C43' => t('Haute-Loire - Auvergne'),
        'C63' => t('Pu-de-Dme - Auvergne'),
        'D21' => t("Cote-d'Or - Bourgogne"),
        'D58' => t('Nievre - Bourgogne'),
        'D71' => t('Saone-et-Loire - Bourgogne'),
        'D89' => t('Yonne - Bourgogne'),
        'E22' => t("Cotes-d'Armor - Bretagne"),
        'E29' => t('Finistere - Bretagne'),
        'E35' => t('Ille-et-Vilaine - Bretagne'),
        'E56' => t('Morbihan - Bretagne'),
        'F18' => t('Cher - Centre'),
        'F28' => t('Eure-et-Loir - Centre'),
        'F36' => t('Indre - Centre'),
        'F37' => t('Indre-et-Loire - Centre'),
        'F41' => t('Loir-et-Cher - Centre'),
        'F45' => t('Loiret - Centre'),
        'G08' => t('Ardennes - Champagne-Ardenne'),
        'G10' => t('Aube - Champagne-Ardenne'),
        'G51' => t('Marne - Champagne-Ardenne'),
        'G52' => t('Haute-Marne - Champagne-Ardenne'),
        'H2A' => t('Corse-du-Sud - Corse'),
        'H2B' => t('Haute-Corse - Corse'),
        'I25' => t('Doubs - Franche-Comte'),
        'I39' => t('Jura - Franche-Comte'),
        'I70' => t('Haute-Saone - Franche-Comte'),
        'I90' => t('Haute-Saone - Territoire de Belfort'),
        'J75' => t('Paris - Ile-de-France'),
        'J77' => t('Seine-et-Marne - Ile-de-France'),
        'J78' => t('Yvelines - Ile-de-France'),
        'J91' => t('Essonne - Ile-de-France'),
        'J92' => t('Hauts-de-Seine - Ile-de-France'),
        'J93' => t('Seine-Saint-Denis - Ile-de-France'),
        'J94' => t('Val-de-Marne - Ile-de-France'),
        'J95' => t("Val-d'Oise - Ile-de-France"),
        'U04' => t('Alpes-de-Haute-Provence'),
        'U05' => t('Hautes-Alpes'),
        'U06' => t('Alpes-Maritimes'),
        'U13' => t('Bouches-du-Rhne'),
        'U83' => t('Var'),
        'U84' => t('Vaucluse'),
      );
      break;
    case 'GE':
      $states[$country] = array(
        'AB' => t('Abkhazia'),
        'AJ' => t('Ajaria'),
        'TB' => t('Tbilisi'),
        'GU' => t('Guria'),
        'IM' => t('Imereti'),
        'KA' => t('Kakheti'),
        'KK' => t('Kvemo Kartli'),
        'MM' => t('Mtskheta-Mtianeti'),
        'RL' => t('Racha Lechkhumi and Kvemo Svaneti'),
        'SZ' => t('Samegrelo-Zemo Svaneti'),
        'SJ' => t('Samtskhe-Javakheti'),
        'SK' => t('Shida Kartli'),
      );
      break;
    case 'IE':
      $states[$country] = array(
        'AM' => t('Antrim'),
        'AH' => t('Armagh'),
        'C'  => t('Cork'),
        'CE' => t('Clare'),
        'CN' => t('Cavan'),
        'CW' => t('Carlow'),
        'D'  => t('Dublin'),
        'DL' => t('Donegal'),
        'DN' => t('Down'),
        'DY' => t('Derry'),
        'FH' => t('Fermanagh'),
        'GW' => t('Galway'),
        'KE' => t('Kildare'),
        'KK' => t('Kilkenny'),
        'KY' => t('Kerry'),
        'LD' => t('Longford'),
        'LH' => t('Louth'),
        'LK' => t('Limerick'),
        'LM' => t('Leitrim'),
        'LS' => t('Laois'),
        'MH' => t('Meath'),
        'MN' => t('Monaghan'),
        'MO' => t('Mayo'),
        'OY' => t('Offaly'),
        'RN' => t('Roscommon'),
        'SO' => t('Sligo'),
        'TN' => t('Tipperary North'),
        'TS' => t('Tipperary South'),
        'TE' => t('Tyrone'),
        'WD' => t('Waterford'),
        'WH' => t('Westmeath'),
        'WW' => t('Wicklow'),
        'WX' => t('Wexford'),
      );
      break;
    case 'IT':
      $states[$country] = array(
        'AG' => t('Agrigento'),
        'AL' => t('Alessandria'),
        'AN' => t('Ancona'),
        'AO' => t('Aosta'),
        'AR' => t('Arezzo'),
        'AP' => t('Ascoli Piceno'),
        'AT' => t('Asti'),
        'AV' => t('Avellino'),
        'BA' => t('Bari'),
        'BL' => t('Belluno'),
        'BN' => t('Benevento'),
        'BG' => t('Bergamo'),
        'BI' => t('Biella'),
        'BO' => t('Bologna'),
        'BZ' => t('Bolzano'),
        'BS' => t('Brescia'),
        'BR' => t('Brindisi'),
        'CA' => t('Cagliari'),
        'CL' => t('Caltanissetta'),
        'CB' => t('Campobasso'),
        'CE' => t('Caserta'),
        'CT' => t('Catania'),
        'CZ' => t('Catanzaro'),
        'CH' => t('Chieti'),
        'CO' => t('Como'),
        'CS' => t('Cosenza'),
        'CR' => t('Cremona'),
        'KR' => t('Crotone'),
        'CN' => t('Cuneo'),
        'EN' => t('Enna'),
        'FE' => t('Ferrara'),
        'FI' => t('Firenze'),
        'FG' => t('Foggia'),
        'FO' => t('Forl�-Cesena'),
        'FR' => t('Frosinone'),
        'GE' => t('Genova'),
        'GO' => t('Gorizia'),
        'GR' => t('Grosseto'),
        'IM' => t('Imperia'),
        'IS' => t('Isernia'),
        'AQ' => t("L'Aquila"),
        'SP' => t('La Spezia'),
        'LT' => t('Latina'),
        'LE' => t('Lecce'),
        'LC' => t('Lecco'),
        'LI' => t('Livorno'),
        'LO' => t('Lodi'),
        'LU' => t('Lucca'),
        'MC' => t('Macerata'),
        'MN' => t('Mantova'),
        'MS' => t('Massa-Carrara'),
        'MT' => t('Matera'),
        'ME' => t('Messina'),
        'MI' => t('Milano'),
        'MO' => t('Modena'),
        'NA' => t('Naploli'),
        'NO' => t('Novara'),
        'NU' => t('Nuoro'),
        'OR' => t('Oristano'),
        'PD' => t('Padova'),
        'PA' => t('Palermo'),
        'PR' => t('Parma'),
        'PV' => t('Pavia'),
        'PG' => t('Perugia'),
        'PS' => t('Pesaro e Urbino'),
        'PE' => t('Pescara'),
        'PC' => t('Piacenza'),
        'PI' => t('Pisa'),
        'PT' => t('Pistoia'),
        'PN' => t('Pordenone'),
        'PZ' => t('Potenza'),
        'PO' => t('Prato'),
        'RG' => t('Ragusa'),
        'RA' => t('Ravenna'),
        'RC' => t('Reggio Calabria'),
        'RE' => t('Reggio Emilia'),
        'RI' => t('Rieti'),
        'RN' => t('Rimini'),
        'RM' => t('Roma'),
        'RO' => t('Rovigo'),
        'SA' => t('Salerno'),
        'SS' => t('Sassari'),
        'SV' => t('Savona'),
        'SI' => t('Siena'),
        'SO' => t('Sondrio'),
        'SR' => t('Siracusa'),
        'TA' => t('Taranto'),
        'TE' => t('Teramo'),
        'TR' => t('Terni'),
        'TP' => t('Trapani'),
        'TN' => t('Trento'),
        'TV' => t('Treviso'),
        'TS' => t('Trieste'),
        'TO' => t('Torino'),
        'UD' => t('Udine'),
        'VA' => t('Varese'),
        'VE' => t('Venezia'),
        'VB' => t('Verbano-Cusio-Ossola'),
        'VC' => t('Vercelli'),
        'VR' => t('Verona'),
        'VV' => t('Vibo Valentia'),
        'VI' => t('Vicenza'),
        'VT' => t('Viterbo'),
      );
      break;
    case 'JP':
      $states[$country] = array(
        '01' => t('Hokkaido'),
        '02' => t('Aomori'),
        '03' => t('Iwate'),
        '04' => t('Miyagi'),
        '05' => t('Akita'),
        '06' => t('Yamagata'),
        '07' => t('Hukusima (Fukushima)'),
        '08' => t('Ibaraki'),
        '09' => t('Totigi (Tochigi)'),
        '10' => t('Gunma'),
        '11' => t('Saitama'),
        '12' => t('Tiba (Chiba)'),
        '13' => t('Tokyo'),
        '14' => t('Kanagawa'),
        '15' => t('Niigata'),
        '16' => t('Toyama'),
        '17' => t('Isikawa (Ishikawa)'),
        '18' => t('Hukui (Fukui)'),
        '19' => t('Yamanasi (Yamanashi)'),
        '20' => t('Nagano'),
        '21' => t('Gihu  (Gifu)'),
        '22' => t('Sizuoka (Shizuoka)'),
        '23' => t('Aiti (Aichi)'),
        '24' => t('Mie'),
        '25' => t('Siga (Shiga)'),
        '26' => t('Kyoto'),
        '27' => t('Osaka'),
        '28' => t('Hyogo'),
        '29' => t('Nara'),
        '30' => t('Wakayama'),
        '31' => t('Tottori'),
        '32' => t('Simane (Shimane)'),
        '33' => t('Okayama'),
        '34' => t('Hirosima (Hiroshima)'),
        '35' => t('Yamaguti (Yamaguchi)'),
        '36' => t('Tokusima (Tokushima)'),
        '37' => t('Kagawa'),
        '38' => t('Ehime'),
        '39' => t('Koti (Kochi)'),
        '40' => t('Hukuoka (Fukuoka)'),
        '41' => t('Saga'),
        '42' => t('Nagasaki'),
        '43' => t('Kumamoto'),
        '44' => t('Oita'),
        '45' => t('Miyazaki'),
        '46' => t('Kagosima (Kagoshima)'),
        '47' => t('Okinawa'),
      );
      break;
    case 'MY':
      $states[$country] = array(
        'JH' => t('Johor'),
        'KD' => t('Kedah'),
        'KN' => t('Kelantan'),
        'KL' => t('Kuala Lumpur'),
        'ML' => t('Malacca'),
        'NS' => t('Negeri Sembilan'),
        'PH' => t('Pahang'),
        'PK' => t('Perak'),
        'PS' => t('Perlis'),
        'PG' => t('Penang'),
        'WP' => t('Wilayah Persekutuan'),
        'SL' => t('Selangor'),
        'TR' => t('Terengganu'),
        'LB' => t('Labuan'),
        'SB' => t('Sabah'),
        'SR' => t('Sarawak'),
      );
      break;
    case 'NL':
      $states[$country] = array(
        'DR' => t('Drenthe'),
        'FL' => t('Flevoland'),
        'FR' => t('Friesland'),
        'GE' => t('Gelderland'),
        'GR' => t('Groningen'),
        'LI' => t('Limburg'),
        'NB' => t('Noord Brabant'),
        'NH' => t('Noord Holland'),
        'OV' => t('Overijssel'),
        'UT' => t('Utrecht'),
        'ZE' => t('Zeeland'),
        'ZH' => t('Zuid Holland'),
      );
      break;
    case 'GB':
    case 'UK':
      $states[$country] = array(
        'GSY' => t('Guernsey'),
        'JSY' => t('Jersey'),
        'BDG' => t('Barking and Dagenham'),
        'BNE' => t('Barnet'),
        'BNS' => t('Barnsley'),
        'BAS' => t('Bath and North East Somerset'),
        'BDF' => t('Bedfordshire'),
        'BEX' => t('Bexley'),
        'BIR' => t('Birmingham'),
        'BBD' => t('Blackburn with Darwen'),
        'BPL' => t('Blackpool'),
        'BOL' => t('Bolton'),
        'BMH' => t('Bournemouth'),
        'BRC' => t('Bracknell Forest'),
        'BRD' => t('Bradford'),
        'BEN' => t('Brent'),
        'BNH' => t('Brighton and Hove'),
        'BST' => t('Bristol City of'),
        'BRY' => t('Bromley'),
        'BKM' => t('Buckinghamshire'),
        'BUR' => t('Bury'),
        'CLD' => t('Calderdale'),
        'CAM' => t('Cambridgeshire'),
        'CMD' => t('Camden'),
        'CHS' => t('Cheshire'),
        'CON' => t('Cornwall'),
        'COV' => t('Coventry (West Midlands district)'),
        'CRY' => t('Croydon'),
        'CMA' => t('Cumbria'),
        'DAL' => t('Darlington'),
        'DER' => t('Derby'),
        'DBY' => t('Derbyshire'),
        'DEV' => t('Devon'),
        'DNC' => t('Doncaster'),
        'DOR' => t('Dorset'),
        'DUD' => t('Dudley (West Midlands district)'),
        'DUR' => t('Durham'),
        'EAL' => t('Ealing'),
        'ERY' => t('East Riding of Yorkshire'),
        'ESX' => t('East Sussex'),
        'ENF' => t('Enfield'),
        'ESS' => t('Essex'),
        'GAT' => t('Gateshead (Tyne & Wear district)'),
        'GLS' => t('Gloucestershire'),
        'GRE' => t('Greenwich'),
        'HCK' => t('Hackney'),
        'HAL' => t('Halton'),
        'HMF' => t('Hammersmith and Fulham'),
        'HAM' => t('Hampshire'),
        'HRY' => t('Haringey'),
        'HRW' => t('Harrow'),
        'HPL' => t('Hartlepool'),
        'HAV' => t('Havering'),
        'HEF' => t('Herefordshire County of'),
        'HRT' => t('Hertfordshire'),
        'HIL' => t('Hillingdon'),
        'HNS' => t('Hounslow'),
        'IOW' => t('Isle of Wight'),
        'IOS' => t('Isles of Scilly'),
        'ISL' => t('Islington'),
        'KEC' => t('Kensington and Chelsea'),
        'KEN' => t('Kent'),
        'KHL' => t('Kingston upon Hull City of'),
        'KTT' => t('Kingston upon Thames'),
        'KIR' => t('Kirklees'),
        'KWL' => t('Knowsley'),
        'LBH' => t('Lambeth'),
        'LAN' => t('Lancashire'),
        'LDS' => t('Leeds'),
        'LCE' => t('Leicester'),
        'LEC' => t('Leicestershire'),
        'LEW' => t('Lewisham'),
        'LIN' => t('Lincolnshire'),
        'LIV' => t('Liverpool'),
        'LND' => t('London City of'),
        'LUT' => t('Luton'),
        'MAN' => t('Manchester'),
        'MDW' => t('Medway'),
        'MRT' => t('Merton'),
        'MDB' => t('Middlesbrough'),
        'MIK' => t('Milton Keynes'),
        'NET' => t('Newcastle upon Tyne'),
        'NWM' => t('Newham'),
        'NFK' => t('Norfolk'),
        'NEL' => t('North East Lincolnshire'),
        'NLN' => t('North Lincolnshire'),
        'NSM' => t('North Somerset'),
        'NTY' => t('North Tyneside'),
        'NYK' => t('North Yorkshire'),
        'NTH' => t('Northamptonshire'),
        'NBL' => t('Northumberland'),
        'NGM' => t('Nottingham'),
        'NTT' => t('Nottinghamshire'),
        'OLD' => t('Oldham'),
        'OXF' => t('Oxfordshire'),
        'PTE' => t('Peterborough'),
        'PLY' => t('Plymouth'),
        'POL' => t('Poole'),
        'POR' => t('Portsmouth'),
        'RDG' => t('Reading'),
        'RDB' => t('Redbridge'),
        'RCC' => t('Redcar and Cleveland'),
        'RIC' => t('Richmond upon Thames'),
        'RCH' => t('Rochdale'),
        'ROT' => t('Rotherham'),
        'RUT' => t('Rutland'),
        'SHN' => t('St Helens'),
        'SLF' => t('Salford'),
        'SAW' => t('Sandwell'),
        'SFT' => t('Sefton'),
        'SHF' => t('Sheffield'),
        'SHR' => t('Shropshire'),
        'SLG' => t('Slough'),
        'SOL' => t('Solihull'),
        'SOM' => t('Somerset'),
        'SGC' => t('South Gloucestershire'),
        'STY' => t('South Tyneside'),
        'STH' => t('Southampton'),
        'SOS' => t('Southend-on-Sea'),
        'SWK' => t('Southwark'),
        'STS' => t('Staffordshire'),
        'SKP' => t('Stockport'),
        'STT' => t('Stockton-on-Tees'),
        'STE' => t('Stoke-on-Trent'),
        'SFK' => t('Suffolk'),
        'SND' => t('Sunderland'),
        'SRY' => t('Surrey'),
        'STN' => t('Sutton'),
        'SWD' => t('Swindon'),
        'TAM' => t('Tameside'),
        'TFW' => t('Telford and Wrekin'),
        'THR' => t('Thurrock'),
        'TOB' => t('Torbay'),
        'TWH' => t('Tower Hamlets'),
        'TRF' => t('Trafford'),
        'WKF' => t('Wakefield'),
        'WLL' => t('Walsall'),
        'WFT' => t('Waltham Forest'),
        'WND' => t('Wandsworth'),
        'WRT' => t('Warrington'),
        'WAR' => t('Warwickshire'),
        'WBK' => t('West Berkshire'),
        'WSX' => t('West Sussex'),
        'WSM' => t('Westminster'),
        'WGN' => t('Wigan'),
        'WIL' => t('Wiltshire'),
        'WNM' => t('Windsor and Maidenhead'),
        'WRL' => t('Wirral'),
        'WOK' => t('Wokingham'),
        'WLV' => t('Wolverhampton'),
        'WOR' => t('Worcestershire'),
        'YOR' => t('York'),
        'ANT' => t('Antrim'),
        'ARD' => t('Ards'),
        'ARM' => t('Armagh'),
        'BLA' => t('Ballymena'),
        'BLY' => t('Ballymoney'),
        'BNB' => t('Banbridge'),
        'BFS' => t('Belfast'),
        'CKF' => t('Carrickfergus'),
        'CSR' => t('Castlereagh'),
        'CLR' => t('Coleraine'),
        'CKT' => t('Cookstown'),
        'CGV' => t('Craigavon'),
        'DRY' => t('Derry'),
        'DOW' => t('Down'),
        'DGN' => t('Dungannon and South Tyrone'),
        'FER' => t('Fermanagh'),
        'LRN' => t('Larne'),
        'LMV' => t('Limavady'),
        'LSB' => t('Lisburn'),
        'MFT' => t('Magherafelt'),
        'MYL' => t('Moyle'),
        'NYM' => t('Newry and Mourne'),
        'NTA' => t('Newtownabbey'),
        'NDN' => t('North Down'),
        'OMH' => t('Omagh'),
        'STB' => t('Strabane'),
        'ABE' => t('Aberdeen'),
        'ABD' => t('Aberdeenshire'),
        'ANS' => t('Angus'),
        'AGB' => t('Argyll and Bute'),
        'CLK' => t('Clackmannanshire'),
        'DGY' => t('Dumfries and Galloway'),
        'DND' => t('Dundee'),
        'EAY' => t('East Ayrshire'),
        'EDU' => t('East Dunbartonshire'),
        'ELN' => t('East Lothian'),
        'ERW' => t('East Renfrewshire'),
        'EDH' => t('Edinburgh'),
        'ELS' => t('Eilean Siar'),
        'FAL' => t('Falkirk'),
        'FIF' => t('Fife'),
        'GLG' => t('Glasgow'),
        'HLD' => t('Highland'),
        'IVC' => t('Inverclyde'),
        'NAY' => t('North Ayrshire'),
        'NLK' => t('North Lanarkshire'),
        'ORK' => t('Orkney Islands'),
        'PKN' => t('Perth and Kinross'),
        'MLN' => t('Midlothian'),
        'MRY' => t('Moray'),
        'RFW' => t('Renfrewshire'),
        'SCB' => t('Scottish Borders The'),
        'ZET' => t('Shetland Islands'),
        'SAY' => t('South Ayrshire'),
        'SLK' => t('South Lanarkshire'),
        'STG' => t('Stirling'),
        'WDU' => t('West Dunbartonshire'),
        'WLN' => t('West Lothian'),
        'BGW' => t('Blaenau Gwent'),
        'BGE' => t('Bridgend'),
        'CAY' => t('Caerphilly'),
        'CRF' => t('Cardiff'),
        'CMN' => t('Carmarthenshire'),
        'CGN' => t('Ceredigion'),
        'CWY' => t('Conwy'),
        'DEN' => t('Denbighshire'),
        'FLN' => t('Flintshire'),
        'GWN' => t('Gwynedd'),
        'AGY' => t('Isle of Anglesey'),
        'MTY' => t('Merthyr Tydfil'),
        'MON' => t('Monmouthshire'),
        'NTL' => t('Neath Port Talbot'),
        'NWP' => t('Newport'),
        'PEM' => t('Pembrokeshire'),
        'POW' => t('Powys'),
        'RCT' => t('Rhondda Cynon Taf'),
        'SWA' => t('Swansea'),
        'TOF' => t('Torfaen'),
        'VGL' => t('Vale of Glamorgan'),
        'WRX' => t('Wrexham'),
      );
      break;
    case 'US':
      $states[$country] = array(
        'AL' => 'Alabama',
        'AK' => 'Alaska',
        'AZ' => 'Arizona',
        'AR' => 'Arkansas',
        'CA' => 'California',
        'CO' => 'Colorado',
        'CT' => 'Connecticut',
        'DE' => 'Delaware',
        'DC' => 'District Of Columbia',
        'FL' => 'Florida',
        'GA' => 'Georgia',
        'HI' => 'Hawaii',
        'ID' => 'Idaho',
        'IL' => 'Illinois',
        'IN' => 'Indiana',
        'IA' => 'Iowa',
        'KS' => 'Kansas',
        'KY' => 'Kentucky',
        'LA' => 'Louisiana',
        'ME' => 'Maine',
        'MD' => 'Maryland',
        'MA' => 'Massachusetts',
        'MI' => 'Michigan',
        'MN' => 'Minnesota',
        'MS' => 'Mississippi',
        'MO' => 'Missouri',
        'MT' => 'Montana',
        'NE' => 'Nebraska',
        'NV' => 'Nevada',
        'NH' => 'New Hampshire',
        'NJ' => 'New Jersey',
        'NM' => 'New Mexico',
        'NY' => 'New York',
        'NC' => 'North Carolina',
        'ND' => 'North Dakota',
        'OH' => 'Ohio',
        'OK' => 'Oklahoma',
        'OR' => 'Oregon',
        'PA' => 'Pennsylvania',
        'RI' => 'Rhode Island',
        'SC' => 'South Carolina',
        'SD' => 'South Dakota',
        'TN' => 'Tennessee',
        'TX' => 'Texas',
        'UT' => 'Utah',
        'VT' => 'Vermont',
        'VA' => 'Virginia',
        'WA' => 'Washington',
        'WV' => 'West Virginia',
        'WI' => 'Wisconsin',
        'WY' => 'Wyoming',
      );
      break;
    default:
      // Look for a custom file that might have a state list. By default place a
      // file store_custom.inc alongside the store module. Or use the settings.php
      // $conf array if you want to define a custom path in your Drupal installation.
      $custom_inc = variable_get('ec_custom_store_inc', drupal_get_path('module', 'ec_store') .'/ec_store_custom.inc');
      if (file_exists($custom_inc)) {
        include_once $custom_inc;
        $states[$country] = ec_store_custom_build_states($country);
      }
  }

  // Return the requested state list.
  if (isset($states[$country])) {
    return $states[$country];
  }
  return array();
}
